"use client"

import { useState, useEffect } from "react"
import { useParams, useRouter } from "next/navigation"
import { Logo } from "@/components/Logo"
import { ThemeToggle } from "@/components/theme-toggle"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Building, Mail, Phone, MapPin, Clock, User, ArrowLeft } from "lucide-react"
import Link from "next/link"

interface Customer {
  id: string
  name: string
  email?: string
  phone?: string
  company?: string
  address?: string
  notes?: string
  createdAt: string
  updatedAt: string
  timeEntries: TimeEntry[]
}

interface TimeEntry {
  id: string
  description: string
  startTime: string
  endTime: string
  duration: number
  createdAt: string
  user: {
    name: string
    email: string
  }
}

export default function CustomerDetail() {
  const params = useParams()
  const router = useRouter()
  const [customer, setCustomer] = useState<Customer | null>(null)
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    if (params.id) {
      fetchCustomer()
    }
  }, [params.id]) // eslint-disable-line react-hooks/exhaustive-deps

  const fetchCustomer = async () => {
    try {
      setLoading(true)
      const response = await fetch(`/api/customers/${params.id}`)
      if (response.ok) {
        const data = await response.json()
        setCustomer(data)
      } else {
        router.push("/kunden")
      }
    } catch (error) {
      console.error("Error fetching customer:", error)
      router.push("/kunden")
    } finally {
      setLoading(false)
    }
  }

  const formatTime = (seconds: number) => {
    const hours = Math.floor(seconds / 3600)
    const minutes = Math.floor((seconds % 3600) / 60)
    return `${hours}h ${minutes}m`
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString("de-DE")
  }

  const totalHours = customer?.timeEntries.reduce((sum, entry) => sum + entry.duration, 0) || 0

  if (loading) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <p className="text-muted-foreground">Lädt...</p>
      </div>
    )
  }

  if (!customer) {
    return null
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="border-b bg-card">
        <div className="container mx-auto flex h-16 items-center justify-between px-4">
          <Logo className="h-8" />
          <div className="flex items-center gap-4">
            <span className="text-sm text-muted-foreground">Mitarbeiter Dashboard</span>
            <ThemeToggle />
          </div>
        </div>
      </header>

      <main className="container mx-auto px-4 py-8 max-w-6xl">
        <div className="mb-8">
          <Link href="/kunden">
            <Button variant="ghost" size="sm" className="mb-4">
              <ArrowLeft className="h-4 w-4 mr-2" />
              Zurück zu Kunden
            </Button>
          </Link>
          <h1 className="text-3xl font-bold mb-2">{customer.name}</h1>
          {customer.company && (
            <p className="text-muted-foreground flex items-center gap-2">
              <Building className="h-4 w-4" />
              {customer.company}
            </p>
          )}
        </div>

        <div className="grid gap-6 md:grid-cols-3">
          {/* Customer Info */}
          <div className="md:col-span-1">
            <Card>
              <CardHeader>
                <CardTitle>Kontaktinformationen</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {customer.email && (
                  <div className="flex items-center gap-3">
                    <Mail className="h-4 w-4 text-muted-foreground" />
                    <div>
                      <div className="text-sm text-muted-foreground">E-Mail</div>
                      <div className="font-medium">{customer.email}</div>
                    </div>
                  </div>
                )}
                {customer.phone && (
                  <div className="flex items-center gap-3">
                    <Phone className="h-4 w-4 text-muted-foreground" />
                    <div>
                      <div className="text-sm text-muted-foreground">Telefon</div>
                      <div className="font-medium">{customer.phone}</div>
                    </div>
                  </div>
                )}
                {customer.address && (
                  <div className="flex items-center gap-3">
                    <MapPin className="h-4 w-4 text-muted-foreground" />
                    <div>
                      <div className="text-sm text-muted-foreground">Adresse</div>
                      <div className="font-medium">{customer.address}</div>
                    </div>
                  </div>
                )}
                {customer.notes && (
                  <div className="pt-4 border-t">
                    <div className="text-sm text-muted-foreground mb-2">Notizen</div>
                    <div className="text-sm whitespace-pre-wrap">{customer.notes}</div>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Statistics */}
            <Card className="mt-6">
              <CardHeader>
                <CardTitle>Statistiken</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div>
                    <div className="text-sm text-muted-foreground">Gesamtzeit</div>
                    <div className="text-2xl font-bold flex items-center gap-2">
                      <Clock className="h-5 w-5" />
                      {formatTime(totalHours)}
                    </div>
                  </div>
                  <div>
                    <div className="text-sm text-muted-foreground">Zeiteinträge</div>
                    <div className="text-2xl font-bold">{customer.timeEntries.length}</div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Time Entries */}
          <div className="md:col-span-2">
            <Card>
              <CardHeader>
                <CardTitle>Zeiterfassung</CardTitle>
                <CardDescription>
                  Alle erfassten Zeiten für diesen Kunden
                </CardDescription>
              </CardHeader>
              <CardContent>
                {customer.timeEntries.length === 0 ? (
                  <p className="text-center text-muted-foreground py-8">
                    Noch keine Zeiteinträge vorhanden
                  </p>
                ) : (
                  <div className="space-y-4">
                    {customer.timeEntries.map((entry) => (
                      <div
                        key={entry.id}
                        className="p-4 border rounded-lg hover:bg-accent/50 transition-colors"
                      >
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <div className="font-medium mb-2">{entry.description}</div>
                            <div className="flex items-center gap-4 text-sm text-muted-foreground">
                              <div className="flex items-center gap-1">
                                <User className="h-3 w-3" />
                                {entry.user.name}
                              </div>
                              <div>{formatDate(entry.createdAt)}</div>
                            </div>
                          </div>
                          <div className="text-lg font-mono font-semibold">
                            {formatTime(entry.duration)}
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </div>
      </main>
    </div>
  )
}

